// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_INOTIFICATION_ENDPOINT_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_INOTIFICATION_ENDPOINT_CALLBACKS_H

#include "common.h"

/**
 * This class represents an interface that every sink implementation must subclass if they
 * wish to implement the generic notification service. You should look at the documentation
 * for class NotificationEndpoint for a pseudocode example on how to set up such an endpoint.
 */
class INotificationEndpointCallbacks {
public:
    virtual ~INotificationEndpointCallbacks() { }
    /**
     * This method is invoked whenever the status of the phones subscription to the
     * notification service changes. If you try to send a notification before the phone
     * has subscribed, it will fail.
     * @param subscribed true if the phone has subscribed to notifications, false if the phone
     *        has unsubscribed.
     */
    virtual void subscriptionStatusCallback(bool subscribed) = 0;
    /**
     * Called when a new notification arrives from the phone.
     * @param text The text of the notification to be displayed.
     * @param hasId If true, the id parameter contains valid data.
     * @param id An identifier for this notification. If an id is present, you must call
     *        ackNotification() with the id once it has been handled.
     * @param hasIcon If true, the icon argument contains valid data.
     * @param icon A pointer to the icon data (png format).
     * @param len The size of the icon in bytes.
     */
    virtual void notificationCallback(const std::string& text, bool hasId,
            const std::string& id, bool hasIcon, uint8_t* icon, size_t len) = 0;
    /**
     * Called when notification is acknowledged by the other end. This will only be called
     * for notifications that explicitly request an acknowledgement.
     * @param id The id of the notification that was acknowledged.
     */
    virtual void ackCallback(const std::string& id) = 0;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_INOTIFICATION_ENDPOINT_CALLBACKS_H
